;**********************************************************************************************
;       UART_TEST.asm - A sample/test main program for Z8 Microcontroller HW & SW UARTS
;                                  Copyright (C) 2000 ZiLOG, Inc
;                                       All rights reserved
;**********************************************************************************************
;	Assembler: ZDS V3.62
;	Best if viewed in Courier New 9 pt with 7 space tabs
;**********************************************************************************************
;
;	                     UARTS FOR Z8 MICROCONTROLLERS
;
;  This program will provide a main loop for demonstrating the usage of the UART.ASM software 
;  and hardware UART routines for the Z8 Microcontrollers.  The program will generate a test
;  output pattern (ASCII characters 0 through ~) on reset, or following receipt of a carrage
;  return character.  Any other character received will be echoed back out.
;
;  This program is intended to operate on the ZiLOG Extreme Exprezzway Z8 SDK, but may be used
;  on any platform as long as I/O definitions match the platform (for the SW UART).
;
;  If using the SDK board ensure that the jumpers are correctly positioned for either the SW or
;  HW UART.  Correct postions of the jumpers are as follows:
;		HW UART (E136 Muze Family):	R16B, R18B	(TxD is P37, RxD is P30)
;		SW UART (any Z8):		R16A, R18A	(TxD is P01, RxD is P32)
;
;**********************************************************************************************
;**********************************************************************************************
;
;  Revision History for UART_TEST.ASM:
;
; 	v1.00 - Production Release							MNG	11/12/00
;**********************************************************************************************

	GLOBALS ON	;For Debugging (so that variables can be symbolically accessed in watch window)

	DEFINE	REG_APP, ORG=%00, SPACE=RFILE	;Application RAM (Starts after I/O Ports) 
	DEFINE	CODE, SPACE=ROM			;Code Space (ROM)

	INCLUDE	"UART.INC" 			;The include file will bring in all externals and
       						;interrupt vectors required for the UART
	
;**************************************************************************************************


	SEGMENT REG_APP 
	
;**************************************************************************************************
; Application Variables
;**************************************************************************************************
		DS	16		; For E136, leave the first bank of working registers unused
					; This is because some of the registers 05h through 0Fh are used by
					; the E136 and cannot be used by an application program
TEST_DATA	DS	1		; Define storage for a single Date Byte variable needed by this program

	SEGMENT	CODE

;*************************************************************************
; Interrupt Vector Table
;*************************************************************************
;
;  Only vectors needed by the main program should be defined here.  Vectors required by the UART are defined
;  in the UART.INC file and overlaid over these default vectors as necessary.
;
;  It is good practice to ensure that all vectors point to a legitimate address (NULLINT), so that any 
;  spurious interrupt will be handled in a deterministic fashion.
;
;  NOTE:  The linker will give a warning when each vector overlay from the UART.INC file is processed. This
;         warning (ZLD-W0003 Warning: Absolute section '$Vector8x' overlaps allocation in 'ROM') will be
;	   eliminated in future ZDS releases and should be ignored.

	VECTOR RESET = SystemReset
	VECTOR IRQ0  = NULLINT 
	VECTOR IRQ1  = NULLINT
	VECTOR IRQ2  = NULLINT
	VECTOR IRQ3  = NULLINT
	VECTOR IRQ4  = NULLINT
	VECTOR IRQ5  = NULLINT	


;**********************************************************************************************
;
;   This MAIN program is simply an application to create test data for the UART Tx and echo any
;   data from the UART Rx to Tx.  A carrage return character on the Rx will cause re-transmission
;   of the test data.
;
;**********************************************************************************************

 
SystemReset:			; ENTRY POINT - System Reset
	LD	SPL,#%80     		; Setup a stack pointer (top of RAM)
	LD	P2M,#%07             ; P20-P22 are inputs for SDK Board
	LD	P2,#%F8		; Set all P2 output pins high 
	LD	P01M,#%04            ; P00-P03 are outputs for SDK Board
	LD	P0,#%FF		; Set all P0 output pins high
	CLR	IRQ
	LD	IPR,#%13
;;;	LD	IMR,#%00 		; If MAIN program used interrupts, enable mask here
;;;	EI				; If MAIN program used interrupts, enable them here
	
	
MAIN:
	call	INIT_UART		; Initalize the UART for operation
MAIN_1:
      	ld	TEST_DATA,#%30	; The test data pattern starts with 30h (ASCII '0')
TX_TEST_DATA_LOOP:
	tm	UART_FLAGS,#TxRDY	; See if Tx is Ready for more data by checking TxRDY flag
       jr	z,TX_TEST_DATA_LOOP	;   If not ready just keep looping (nothing else to do)
      	ld	TxDATA,TEST_DATA	;   If ready then send the test data to the UART Tx via TxDATA
       CALL	SEND_BYTE		; Call the routine to actually send the byte in the TxDATA interface
	inc	TEST_DATA    		; Generate the next character in the test data set
	CP	TEST_DATA,#%7F	; The test data set ends at 7Fh, the last of the printable ASCII chars
	jr	nz,TX_TEST_DATA_LOOP	; If more test data stay in this loop

ECHO_LOOP:	; If test data loop completed, go to a data echo loop
		; In this echo loop any Rx character will be echoed out the Tx
		; To exit the loop a <CR> character must be received
	
	tm	UART_FLAGS,#RxRDY	; See if Rx data has come in yet
	jr	z,ECHO_LOOP
	and	UART_FLAGS,#~RxRDY	; Clear the RxRDY flag

	CP	RxDATA,#%0D		; Was the Rx character a <CR>?
	jr	z,MAIN_1		; If so, then exit the echo loop
	ld	TxDATA,RxDATA		; If not then echo the Rx data to Tx
	CALL	SEND_BYTE		
	jr	ECHO_LOOP
	
; Make sure that any spurious interrupts don't cause the machine to hang
; Give them returns, just in case

NULLINT:
     	IRET            

;********************************** END OF UART_TEST MAIN PROGRAM ***************************************


